<?php
// Set agar skrip tidak timeout dan berjalan dari direktori yang benar
set_time_limit(0);
chdir(__DIR__);

// Memanggil file konfigurasi utama
require_once 'config.php';
$log_file = __DIR__ . '/cron.log';

/**
 * Fungsi untuk menulis log ke file teks.
 * Berguna untuk debugging manual di server.
 * @param string $message Pesan log.
 */
function writeToLog($message) {
    global $log_file;
    $log_entry = "[" . date('Y-m-d H:i:s') . "] " . $message . PHP_EOL;
    file_put_contents($log_file, $log_entry, FILE_APPEND);
}

/**
 * Fungsi untuk mengirim notifikasi ke Telegram.
 * @param string $pesan Isi pesan.
 * @param string $botToken Token bot dari config.
 * @param string $chatId ID chat tujuan.
 * @param int $jumlahKirim Berapa kali pesan akan di-spam.
 */
function kirimNotifikasi($pesan, $botToken, $chatId, $jumlahKirim = 1) {
    if (empty($botToken) || empty($chatId)) { return; }
    for ($i = 0; $i < $jumlahKirim; $i++) {
        $urlApi = "https://api.telegram.org/bot{$botToken}/sendMessage";
        $postData = ['chat_id' => $chatId, 'text' => $pesan, 'parse_mode' => 'HTML'];
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $urlApi);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_exec($ch);
        curl_close($ch);
        if ($jumlahKirim > 1) usleep(250000); // Jeda 0.25 detik antar pesan spam
    }
}

/**
 * Fungsi untuk mengecek domain via layanan relay.
 * @param string $domain Domain yang akan dicek.
 * @return array|null Data JSON dari relay atau null jika gagal.
 */
function checkDomainViaRelay($domain) {
    $relay_url = "http://103.181.182.57/relay.php?domain=" . urlencode($domain);
    $options = ["http" => ["timeout" => 20, "ignore_errors" => true]];
    $context = stream_context_create($options);
    $response_json = @file_get_contents($relay_url, false, $context);
    if ($response_json === false) { return null; }
    $data = json_decode($response_json, true);
    return (json_last_error() === JSON_ERROR_NONE && !empty($data)) ? $data : null;
}

// =================================================================
// PROSES UTAMA CRON JOB (VERSI STATEFUL DENGAN NOTIFIKASI BERULANG)
// =================================================================

writeToLog("=== Cron Job Dimulai ===");
logActivity("Cron Job pengecekan domain dimulai.");

$query = "SELECT d.id as domain_id, d.name as domain_name, d.url as domain_url, d.is_under_alert, u.id as user_id, u.telegram_chat_id FROM domains d JOIN users u ON d.user_id = u.id WHERE d.is_active = 1";
$result = $conn->query($query);

if ($result->num_rows === 0) {
    writeToLog("Tidak ada domain aktif untuk dicek. Proses selesai.");
    logActivity("Cron Job selesai: Tidak ada domain aktif.");
    $conn->close(); // Tutup koneksi di sini juga jika tidak ada pekerjaan
    exit;
}

writeToLog("Ditemukan " . $result->num_rows . " domain aktif untuk diproses.");

// Loop utama untuk memproses setiap domain satu per satu
while ($domain_data = $result->fetch_assoc()) {
    $domain_id = $domain_data['domain_id'];
    $domain_name = $domain_data['domain_name'];
    $domain_url = $domain_data['domain_url'];
    $is_currently_under_alert = (int)$domain_data['is_under_alert'];
    $domain_owner_id = $domain_data['user_id'];

    writeToLog("Mengecek: {$domain_url} (ID: {$domain_id}, Status Alert Saat Ini: {$is_currently_under_alert})");
    $relay_response = checkDomainViaRelay($domain_url);

    if ($relay_response === null || !isset($relay_response['status'])) {
        writeToLog("   -> Gagal mendapatkan data dari relay. Melewati domain ini.");
        continue;
    }

    $status_from_relay = $relay_response['status'];
    $new_status_db = (trim(strtolower($status_from_relay)) === "ada") ? "BLOCKED" : "SAFE";

    if ($new_status_db === 'BLOCKED') {
        if ($is_currently_under_alert === 0) {
            writeToLog("   -> [AKSI] BARU TERDETEKSI BLOCKED! Mengirim notifikasi utama dan set status alert.");
            $conn->query("UPDATE domains SET is_under_alert = 1 WHERE id = $domain_id");
            if (!empty($domain_data['telegram_chat_id'])) {
                $waktu_deteksi = date('d-m-Y H:i:s');
                $pesan  = "🔥 <b>BLOCKED by Kominfo (Deteksi Pertama)</b> 🔥\n\n";
                $pesan .= "<b>Website:</b> " . e($domain_name) . "\n";
                $pesan .= "<b>URL:</b> <code>" . e($domain_url) . "</code>\n";
                $pesan .= "<b>Input:</b> " . $waktu_deteksi . "\n\n";
                $pesan .= "Anda dapat mengganti website dengan perintah:\n";
                $pesan .= "<code>/replace " . e($domain_url) . " domain_baru.com</code>";
                kirimNotifikasi($pesan, TELEGRAM_BOT_TOKEN, $domain_data['telegram_chat_id'], 5);
                logActivity("Notifikasi BLOCKED (pertama) dikirim untuk domain: $domain_url", $domain_owner_id);
            }
        } else {
            writeToLog("   -> [AKSI] MASIH BLOCKED. Mengirim notifikasi pengingat.");
            if (!empty($domain_data['telegram_chat_id'])) {
                $pesan  = "⚠️ <b>PERINGATAN: WEBSITE MASIH TERBLOKIR</b> ⚠️\n\n";
                $pesan .= "<b>Website:</b> " . e($domain_name) . "\n";
                $pesan .= "<b>URL:</b> <code>" . e($domain_url) . "</code>\n\n";
                $pesan .= "Segera ganti domain Anda untuk menghindari gangguan lebih lanjut.";
                kirimNotifikasi($pesan, TELEGRAM_BOT_TOKEN, $domain_data['telegram_chat_id'], 5);
                logActivity("Notifikasi PENGINGAT BLOCKED dikirim untuk domain: $domain_url", $domain_owner_id);
            }
        }
    } else { // new_status_db === 'SAFE'
        if ($is_currently_under_alert === 1) {
            writeToLog("   -> [AKSI] MASALAH SELESAI. Mereset status alert menjadi 0.");
            $conn->query("UPDATE domains SET is_under_alert = 0 WHERE id = $domain_id");
            logActivity("Status domain $domain_url kembali SAFE.", $domain_owner_id);
        } else {
            writeToLog("   -> Status tetap aman. Tidak ada aksi.");
        }
    }

    foreach ($providers as $provider) {
        $stmt_update = $conn->prepare("INSERT INTO domain_statuses (domain_id, provider, status, last_checked) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE status = VALUES(status), last_checked = NOW()");
        $stmt_update->bind_param("iss", $domain_id, $provider, $new_status_db);
        $stmt_update->execute();
        $stmt_update->close();
    }
    
    sleep(1);
}

// ========================================================
//          PERBAIKAN ADA DI BAGIAN INI
// ========================================================
// Log terakhir dicatat SEBELUM koneksi ditutup.
writeToLog("=== Cron Job Selesai ===");
logActivity("Cron Job pengecekan domain selesai.");

// Koneksi ditutup sebagai LANGKAH PALING AKHIR.
$conn->close();
// ========================================================

?>